module mod_init
! SYMOS ČHMÚ. Program pro výpočet rozptylu znečišťujících látek podle 
! Metodické příručky k modelu SYMOS’97 - aktualizace 2013 (Příloha 1 
! Metodického pokynu odboru ochrany ovzduší MŽP ke zpracování rozptylových 
! studií viz <http://www.mzp.cz/cz/zpracovani_rozptylovych_studii_metodika>. 
! Odlišnosti od Metodické příručky jsou uvedeny v Uživatelské dokumentaci
! modelu.
!
! Copyright (C) 2016 Český hydrometeorologický ústav; 
! Kontaktní osoby (pouze pro zasílání připomínek; ČHMÚ neposkytuje technickou
! podporu): 
!     Nina Benešová <nina.benesova@chmi.cz>, 
!     Ondřej Vlček  <ondrej.vlcek@chmi.cz>
!
! Tento program je svobodným softwarem: můžete jej používat, šířit a 
! upravovat podle ustanovení Obecné veřejné licence GNU (GNU General Public 
! Licence), vydávané Free Software Foundation, a to na základě 3. verze 
! této Licence, nebo (podle vašeho uvážení) kterékoli pozdější verze.
!
! Tento program je rozšiřován v naději, že bude užitečný. Ačkoliv před 
! uvolněním pro veřejnost prochází pečlivým testováním, NEMUSÍ BÝT ZARUČENÁ 
! JEHO PLNÁ FUNKČNOST A STAŽENÍM ZA NĚJ NEJSOU POSKYTOVÁNY JAKÉKOLIV ZÁRUKY 
! VČ. ZÁRUKY ZA VADY a to i tehdy, pokud rozhodným právem nebude právo 
! České republiky.  Při spuštění jakékoliv jeho verze vč. aktualizací není 
! možné požadovat jakékoliv náhrady škody, pokud vznikají jeho užíváním, 
! šířením a úpravou či nesprávným instalováním na jakémkoliv zařízení. 
! Nejsou poskytovány ani odvozené záruky PRODEJNOSTI anebo VHODNOSTI PRO 
! URČITÝ ÚČEL. 
!
! Další podrobnosti hledejte v Obecné veřejné licenci GNU, jejíž kopii jste 
! měli obdržet spolu s tímto programem. Pokud se tak nestalo, najdete ji zde:
! http://www.gnu.org/licenses/gpl.html.
!
! Veškeré práce využívající tento program musí obsahovat následující citaci: 
! „ČHMÚ. SYMOS ČHMÚ v<číslo verze> [software]. [přístup <datum>]. 
!  Dostupné z: <odkaz na www>. K dispozici pro Windows a Linux.“ 
! Upravené verze programu musí být náležitě označeny a obsahovat popis
! provedených změn.

 !
 !> Modul s uživatelem definovanými proměnnými. Definice typů a subroutiny zajišťující načítání vstupních dat.
 !
 implicit none

 integer, parameter	:: dp = kind(0.d0)				!< dvojitá přesnost
 integer		:: trida_lat						!< třída látky

 ! meterologicke parametry
 real(dp)				  :: u_height = 10						!< výška, ve které je zadán vítr [m]
 real(dp), dimension(:,:), allocatable	  :: w_dir            	!< směr větru (čas. průběh) [\f$ ^\circ \f$]
 real(dp), dimension(:,:), allocatable	  :: w_sp            	!< rychlost větru (čas. průběh) [m/s]
 real(dp), dimension(:,:), allocatable	  :: tep_gr           	!< teplotní gradient (čas. průběh) [\f$ ^\circ \f$C/100 m]
 integer,  dimension(:,:), allocatable    :: tridy_st    	!< třída stability (čas. průběh) [-]
 real(dp), dimension(:,:), allocatable	  :: w_dir_i    	!< směr větru nad inverzí (čas. průběh) [\f$ ^\circ \f$]
 real(dp), dimension(:,:), allocatable	  :: w_sp_i      	!< rychlost větru nad inverzí (čas. průběh) [m/s]
 real(dp), dimension(:,:), allocatable	  :: tep_gr_i    	!< teplotní gradient nad inverzí (čas. průběh) [\f$ ^\circ \f$C/100 m]
 integer,  dimension(:,:), allocatable    :: tridy_st_i  	!< třída stability (čas. průběh)
 real(dp), dimension(:,:), allocatable	  :: zi          	!< výška inverze (čas. průběh) [m]
 character(13), dimension(:), allocatable :: date               !< pole s řetězci datum/hodina 
 character(3),  dimension(1000)           :: met_name           !< pole s názvy vstupních hodinových dat
 character(3),  dimension(1000)           :: time_prof_name     !< pole s názvy časových profilů emisí
 integer				  :: met_num = 0 	!< počet souborů se vstupními hodinovými daty 
 integer				  :: time_prof_num = 0 	!< počet souborů s časovými profily emisí 

 ! dalsi casove parametry (emise)
 real(dp), dimension(:,:), allocatable	  ::  time_emis_fact    !< pole s časovým emisním faktorem

 ! ruzice
 integer, parameter				:: poc_rozpt_t = 11				!< počet tříd stability, počet směrů růžice
 real(dp), dimension(:,:,:), allocatable	:: ruzice	        !< větrná růžice zadaná uživatelem
 real(dp)					:: phi_step = 1						!< krok pro rozpočet růžice
 integer				        :: nphi							!< počet kroků přes azimut větru

 ! promenne souvisejici s vypoctem chladicich vezi
 integer, parameter		:: num_te = 10						   !< počet třídních teplot
 integer, parameter		:: num_hum = 7						   !< počet třídních vlhkostí vzduchu
 integer, parameter		:: poc_trid_st = 5					   !< počet tříd stability (definováno i v mod_funkce)
 integer, parameter		:: poc_trid_u = 3					   !< počet třídních rychlostí (definováno i v mod_funkce)
 real(dp), parameter            :: t_t(num_te) = (/-12, -7, -2, 3, 8, 13, 18, 23, 28, 33/) !< třídní teploty [\f$ ^\circ \f$C]
 real(dp), parameter            :: r_t(num_hum) = (/40, 60, 75, 83, 88, 93, 98/)     	   !< třídní vlhkosti vzduchu [%]
 real(dp)			:: grt(num_hum, num_te, poc_trid_st)			   !< relativní četnosti výskytu situace s teplotou t a vlhkostí r
 real(dp)			:: frt(num_hum, num_te)					   !< průměrné relativní množství zkondenzované páry
 real(dp)			:: Vch1(num_hum, num_te, poc_trid_u)     	           !< objemový tok vzduchu bez zavedených spalin [m\f$^{3}\f$ s \f$^{-1}\f$]
 real(dp)			:: Tch1(num_hum, num_te, poc_trid_u)     	       	   !< teplota vzduchu odchazející z věže bez zavedených spalin [\f$ ^\circ \f$C]
  
 ! nazvy vstupnich souboru
 character(250)			:: iputDir = './', oputDir = './'
 character(100)			:: ref_body_file = '', zdroje_file = '', zdroje_popis_file = '', time_file = '', output_ref_p = ''
 character(100)                 :: vyskopis_file = '', ruzice_file = '', output_stats = '', output_time = '', chl_v_file = ''
 character(100)                 :: emis_time_file = ''

 ! parametry vypoctu
 logical		:: is_tep_gr = .TRUE.		!< TRUE pokud na vstupu je tep. gradient, FALSE pokud třída stability
 logical 		:: is_time = .FALSE.		!< TRUE pokud se budou počítat hodinové řady
 logical 		:: is_average = .FALSE.		!< TRUE pokud se budou počítat průměry
 logical 		:: is_max_abs = .FALSE.		!< TRUE pokud se bude počítat absolutní maximum
 logical		:: is_max_11 = .FALSE.		!< TRUE pokud se budou počítat maxima pro 11 rozptylovych tříd
 logical		:: is_daily = .FALSE.		!< TRUE pokud se budou počítat i denní statistiky
 logical		:: is_exceed = .FALSE.		!< TRUE pokud se budou počítat doby překročení
 logical		:: is_no = .FALSE.			!< TRUE pokud se bude počítat konverze NO na NO2
 logical		:: is_shluky = .FALSE.		!< TRUE pokud se bude počítat se shluky
 logical		:: free_atmo = .FALSE.		!< TRUE pokud se bude přímý rozptyl počítat jako ve volné atmosféře
 logical		:: wind_rotation = .TRUE.	!< TRUE pokud se bude uvažovat stáčení směru větru s výškou
 logical		:: inverze = .FALSE.		!< TRUE pokud budeme zohledňovat inverzní vrstvu do vypočtu
 logical 		:: time_output_all = .TRUE.     !< TRUE pokud chci ve výstupním hodinovém texťáku vsechny ref. body
 character(8)		:: tep_vyd_type = 'vyzk-zpr'	!< určuje způsob výpočtu tep. vyd. Q; buď dle metodiky nebo dle výzkumné zprávy
 real(dp)		:: infl_dist_min = 0    	!< vzdálenost od které mají zdroje vliv [m]
 real(dp)		:: infl_dist_max = 100000	!< vzdálenost do které mají zdroje vliv [m]

 integer                :: shortterm_avg = 60           !< [min]   cas prumerovani kratkodobych koncentraci v min
 character(10)		:: delim_out = ';'              !< oddelovac sloupcu ve vystupnim souboru

 character(4)		:: daily_type                   !< dle jakeho typu se budou počítat denní maxima? 

 integer, parameter	:: num_lim = 3                                  !< počet limitů překročení
 real(dp)		:: ch_exceed(num_lim), cd_exceed(num_lim)	!< hodinové a denní limity pro překročení	

 ! vyskopis
 integer		:: ncols, nrows                 !< počet sloupců/řádků výškopisu
 real(dp)		:: xllcorner, yllcorner, xllcorner_upper, yllcorner_upper !< souřadnice levého dolního a pravého horního rohu
 real(dp)		:: cellsize, nodata_value 	!< rozměr buňky, hodnota pro nezadanou výšku
 integer(2), dimension(:,:), allocatable	:: z 	!< výškopis
 real(dp)		:: int_err = 25                 !< přesnost výpočtu integrálu

 type ref_bod                               !< definice typu pro uchování parametrů referenčního bodu
     character(20)	:: id		 			!< id ref. bodu
     real(dp)		:: x, y                 !< souřadnice referenčního bodu v zákl. souř. systemů (\f$x_{r}\f$, \f$y_{r}\f$)   
     real(dp)		:: z           	        !< nadmořská výška v místě ref. bodu (\f$z_{r}\f$)
     real(dp)		:: l	                !< výška ref. bodu nad úrovní terénu (\f$l\f$)
     logical		:: output = .TRUE.      !< .TRUE. pokud má být bod ve výpisu pro celý časový úsek
 end type ref_bod 

 type zdroj								!< definice typu pro uchování parametrů zdroje
     character(20)	:: id				!< id zdroje
     real(dp)		:: x, y          	!< souřadnice zdroje v zákl. souř. systému (\f$x_{z}\f$, \f$y_{z}\f$) (střed u liniového zdroje)  
     real(dp)		:: x1, y1          	!< začátek elementu u liniového zdroje
     real(dp)		:: x2, y2          	!< konec elementu u liniového zdroje 
     real(dp)		:: z				!< nadmořská výška v místě zdroje (\f$z_{z}\f$)
     integer		:: typ				!< typ zdroje - bodový (1), plošný (2), liniový (3)
     integer		:: shluk, skupina       !< id shluku/skupiny
     real(dp)		:: Mz                   !< emise (\f$M_{z}\f$) 
     real(dp)		:: Mz_NO                !< emise NO pro výpočet konverze, bere se v potaz pouze pokud is_no = .true.
     real(dp)		:: emis_fact			!< faktor, kterým se přenásobí zadané emise (typicky pro zadání časového chodu)
     real(dp)		:: H				!< stavební výška zdroje (\f$H\f$)
     real(dp)		:: Vs               !< objemový tok spalin za norm. podm. (\f$V_{s}\f$)
     real(dp)		:: ts              	!< teplota spalin v koruně komína (\f$t_{s}\f$)
     real(dp)		:: d			!< vnitřní průměr výduchu (\f$d\f$)
     real(dp)		:: w0			!< vystupní rychlost exhalací (\f$w_{0}\f$)
     real(dp)		:: alfa			!< relativní roční využítí maximálního výkonu (\f$ alfa\f$)
     real(dp)		:: Pd			!< počet hodin za den kdy je zdroj v provozu
     real(dp)		:: beta			!< parametr beta pro určování prevýšení vlečky (\f$ beta\f$)
     real(dp)		:: A, B			!< parametr A, B pro určování prevýšení vlečky (\f$ A\f$, \f$ B\f$)
     real(dp)		:: Q			!< tepelná vydatnost (\f$ Q \f$)			
     real(dp)		:: y0			!< délka elementu (jen plošné a liniové) 
     real(dp)		:: x0			!< šířka silnice (jen liniové) 
     real(dp)		:: z0			!< výška do které se přízemní exhalace dostanou turbulencí (jen liniové) 
     integer		:: met			!< index meteorologické řady, kterou použít pro tento zdroj
     integer		:: time_prof	!< index časového profilu emisí, který použít pro tento zdroj
     logical		:: chl_v		!< je zdroj chladící věž?
 end type zdroj

 type(ref_bod), dimension(:), allocatable	:: ref_body
 type(zdroj),   dimension(:), allocatable	:: zdroje

namelist/init/ iputDir, ref_body_file, zdroje_file, zdroje_popis_file, time_file, emis_time_file, output_time, vyskopis_file, &
               trida_lat, is_tep_gr, is_average, ruzice_file, is_time, output_stats, oputDir, is_exceed, ch_exceed, cd_exceed, &
               is_no, u_height, daily_type, phi_step, chl_v_file, inverze, free_atmo, is_shluky, is_max_abs, is_max_11, &
               is_daily, time_output_all, output_ref_p, wind_rotation, shortterm_avg, delim_out, int_err, tep_vyd_type, &
               infl_dist_min, infl_dist_max

private :: dp, init, ref_body_file, zdroje_file, zdroje_popis_file, time_file, vyskopis_file, &
           ruzice_file, emis_time_file, chl_v_file, output_ref_p, ncols, nrows, nodata_value, tep_vyd_type

contains



! -------------------------------------------------------------------------------------------------------------------------------
!> Načte uživatelské parametry výpočtu ze souboru nmlFile
!> -------------------------
!> zavolá další subroutiny a načte všechna potřebná data (meteorologie, referenční body, zdroje...) \n
!> ierror kódy: 
!>    -# 0 - vše OK 
!>    -# 1 - požadovaný soubor nebyl zadán
!>    -# 2 - soubor (nebo adresar) neexistuje 
!>    -# 3 - neočekávaný obsah souboru (chybí data nebo jsou jiného typu)
!>    -# 4 - ref. bod/zdroj se nachází mimo síť
!>    -# 5 - neočekávaná hodnota parametru v souboru nmlFile
!>    -# -1 - zdroj byl definován, ale chybí popis

subroutine read_input_data(nmlFile)

 implicit none

 character(*), intent(in   ) 	:: nmlFile           !< nml soubor s uživatelskými parametry
 integer			:: nmlUnit, ierror, ios
 character(400)			:: error_msg

 write(*,*) 'Nacitam vstupni data'  
 ! nacte parametry vypoctu
 error_msg = 'inicializaci vypoctu'
 call check_file_exist(nmlFile, ierror, error_msg)
 if (ierror /= 0) call raise_error(error_msg)
 open(unit = newunit(nmlUnit), file = nmlFile, delim = 'apostrophe')
  read(nmlUnit, nml = init, iostat = ios)  
  if (ios /= 0) then
     call error_nml_read(nmlFile, ios, ierror, error_msg)
     call raise_error(error_msg)
  endif
 close(nmlUnit) 

 ! dopln lomitka na konec cest
 call add_sep2dir(iputDir)
 call add_sep2dir(oputDir)

 ! pocet kroku pres azimut vetru
 nphi = int(360 / phi_step)

 ! nacte data 
 call read_terrain_definition( trim(iputDir) // trim(vyskopis_file), ierror, error_msg )
   if (ierror /= 0) call raise_error(error_msg)
 call read_ref_points( trim(iputDir) // trim(ref_body_file), ierror, error_msg )
   if (ierror /= 0) call raise_error(error_msg)
 call read_sources( trim(iputDir) // trim(zdroje_file), trim(iputDir) // trim(zdroje_popis_file), ierror, error_msg )
   if (ierror /= 0) call raise_error(error_msg)
 write(*, '(A24)', advance = 'no') 'nacitam vyskopis ... '
 call read_terrain( trim(iputDir) // trim(vyskopis_file), ierror, error_msg )
   if (ierror /= 0) call raise_error(error_msg)
 write(*,*) 'OK'
 call dopln_z
 if (is_average .or. is_exceed) call read_ruzice( trim(iputDir) // trim(ruzice_file), ierror, error_msg )
   if (ierror /= 0) call raise_error(error_msg)
 if (is_time) then 
   call read_time( trim(iputDir) // trim(time_file), ierror, error_msg )
   if (ierror /= 0) call raise_error(error_msg)
   call read_time_emis( trim(iputDir) // trim(emis_time_file), ierror, error_msg )
   if (ierror /= 0) call raise_error(error_msg)
 endif
 if (any(zdroje%chl_v)) call read_chlad_vez( trim(iputDir) // trim(chl_v_file), ierror, error_msg ) 
   if (ierror /= 0) call raise_error(error_msg)
 if ((is_time) .and. (.not.time_output_all)) call read_output_ref_p( trim(iputDir) // trim(output_ref_p), ierror, error_msg ) 
   if (ierror /= 0) call raise_error(error_msg)

 ! zkontroluje existenci adresare pro vystupy
 error_msg = 'adresare pro vystupni data'
 call check_directory_exist(trim(oputDir), ierror, error_msg) 
   if (ierror /= 0) call raise_error(error_msg)

 ! kontrola zda byly zadany nazvy vystupnich souboru
 if (is_time) then
    if (output_time == '') then
       ierror = 2
       error_msg = 'Název souboru s hodinovými výstupy (output_time) nebyl zadán.'
       if (ierror /= 0) call raise_error(error_msg)
    endif
 endif

 if (is_max_abs .or. is_average .or. is_max_11 .or. is_exceed) then
    if (output_stats == '') then
       ierror = 2
       error_msg = 'Název souboru pro výstup výpočtu dle metodiky (output_stats) nebyl zadán.'
       if (ierror /= 0) call raise_error(error_msg)
    endif
 endif

end subroutine read_input_data




! -------------------------------------------------------------------------------------------------------------------------------
!> Načte hodinové meteorologické vstupy 
!> -------------------------
!> ze souboru time_file načte data a alokuje příslušné proměnné 

subroutine read_time(path_file, ierror, error_msg)

 implicit none

 character(*), intent(in   )	:: path_file       !< vstupní soubor (i s cestou)
 integer,      intent(  out)    :: ierror          !< chybový kód
 character(*), intent(  out)	:: error_msg       !< chybová zpráva
 character(200)			:: path_file_name
 character(200)			:: head
 integer			:: timeUnit, num_rows, ios, row, y, m, d, h, ind, met 
 real(dp)			:: dir, speed, stab	
 real(dp)			:: dir_i, speed_i, stab_i, h_i

 ! nejdrive projde prvni soubor a spocte pocet radku
 ind = index(path_file, '.', back = .true.)
 if (trim(met_name(1)) == '') then
    path_file_name = path_file
 else
    path_file_name = path_file(:ind-1) // '_' // trim(met_name(1)) // path_file(ind:)
 endif

 error_msg = 'meteorologickou casovou radou'
 call check_file_exist(trim(path_file_name), ierror, error_msg)
 if (ierror /= 0) return

 num_rows = 0
 open(unit = newunit(timeUnit), file = trim(path_file_name), action = 'read')
  read(timeUnit, * , iostat = ios) head;  head = trim(head)
  do while (head(1:1) == '!')
      read(timeUnit, * , iostat = ios) head;  head = trim(head)
  enddo
  backspace(timeUnit)

 do 
     read(timeUnit, * , iostat = ios) y
     if (ios < 0) exit
     num_rows = num_rows + 1
 enddo
 close(timeUnit)

 ! dle poctu radku alokuje promenne
 if (allocated(w_dir)) deallocate(w_dir); allocate(w_dir(met_num, num_rows))
 if (allocated(w_sp)) deallocate(w_sp); allocate(w_sp(met_num, num_rows))
 if (allocated(tep_gr)) deallocate(tep_gr); allocate(tep_gr(met_num, num_rows))
 if (allocated(tridy_st)) deallocate(tridy_st); allocate(tridy_st(met_num, num_rows))
 if (allocated(date)) deallocate(date); allocate(date(num_rows))
 
 if (inverze) then
  if (allocated(w_dir_i)) deallocate(w_dir_i); allocate(w_dir_i(met_num, num_rows))
  if (allocated(w_sp_i)) deallocate(w_sp_i); allocate(w_sp_i(met_num, num_rows))
  if (allocated(tep_gr_i)) deallocate(tep_gr_i); allocate(tep_gr_i(met_num, num_rows))
  if (allocated(tridy_st_i)) deallocate(tridy_st_i); allocate(tridy_st_i(met_num, num_rows))
  if (allocated(zi)) deallocate(zi); allocate(zi(met_num, num_rows))
 end if
 
 ! projde vsechny soubory a nacte hodnoty
 do met = 1, met_num
    if (trim(met_name(met)) == '') then
       path_file_name = path_file
    else
       path_file_name = path_file(:ind-1) // '_' // trim(met_name(met)) // path_file(ind:)
    endif
    call check_file_exist(trim(path_file_name), ierror, error_msg)
    if (ierror /= 0) return
    open(unit = newunit(timeUnit), file = trim(path_file_name), action = 'read')
     read(timeUnit, * , iostat = ios) head;  head = trim(head)
     do while (head(1:1) == '!')
         read(timeUnit, * , iostat = ios) head;  head = trim(head)
     enddo
     backspace(timeUnit)

    do row = 1, num_rows
        if (inverze) then
           read(timeUnit, * , iostat = ios) y, m, d, h, dir, speed, stab, h_i, dir_i, speed_i, stab_i
        else
           read(timeUnit, * , iostat = ios) y, m, d, h, dir, speed, stab
        endif
        if (ios /= 0) then
           call error_reading(path_file_name, row, ios, ierror, error_msg)
           close(timeUnit)
           return
        endif

        write(date(row), '(I4, A, I0.2, A, I0.2, A, I0.2)') y, '/', m, '/', d, '/', h

        w_dir(met, row) = dir
        if (speed < 1.5) speed = 1.5_dp   
        w_sp(met, row) = speed 
        if (is_tep_gr) then
           tep_gr(met, row) = stab
        else
           tridy_st(met, row) = int(stab)
        endif
        if (inverze) then
           w_dir_i(met, row) = dir_i
           if (speed_i < 1.5) speed_i = 1.5_dp 
           w_sp_i(met, row) = speed_i
           zi(met, row) = h_i
           if (is_tep_gr) then
	      tep_gr_i(met, row) = stab_i
           else
	      tridy_st_i(met, row) = int(stab_i)
           endif     
        end if
    enddo
    close(timeUnit)
 enddo

end subroutine read_time





! -------------------------------------------------------------------------------------------------------------------------------
!> Načte časové variace emisí 
!> -------------------------
!> ze souboru emis_time_file načte data a alokuje příslušné proměnné 

subroutine read_time_emis(path_file, ierror, error_msg)

 implicit none

 character(*), intent(in   )	:: path_file       !< vstupní soubor (i s cestou)
 integer,      intent(  out)    :: ierror          !< chybový kód
 character(*), intent(  out)	:: error_msg       !< chybová zpráva
 character(200)			:: path_file_name
 character(200)			:: head
 integer			:: emisUnit, num_rows, ios, row, y, m, d, h, ind, time_prof 
 real(dp)			:: fact


 error_msg = 'emisni casovou radou'
 num_rows = size(w_dir, 2)
 if (allocated(time_emis_fact)) deallocate(time_emis_fact); allocate(time_emis_fact(time_prof_num, num_rows))

 ! nothing to do if no time emis dependence
 if (emis_time_file == '') then
    time_emis_fact = 1
    return
 endif
 
 ! projde vsechny soubory a nacte hodnoty
 ind = index(path_file, '.', back = .true.)
 do time_prof = 1, time_prof_num
    if (trim(time_prof_name(time_prof)) == '') then
       path_file_name = path_file
    else
       path_file_name = path_file(:ind-1) // '_' // trim(time_prof_name(time_prof)) // path_file(ind:)
    endif
    call check_file_exist(trim(path_file_name), ierror, error_msg)
    if (ierror /= 0) return
    open(unit = newunit(emisUnit), file = trim(path_file_name), action = 'read')
     read(emisUnit, * , iostat = ios) head;  head = trim(head)
     do while (head(1:1) == '!')
         read(emisUnit, * , iostat = ios) head;  head = trim(head)
     enddo
     backspace(emisUnit)

    do row = 1, num_rows
        read(emisUnit, * , iostat = ios) y, m, d, h, fact
        if (ios /= 0) then
           call error_reading(path_file_name, row, ios, ierror, error_msg)
           close(emisUnit)
           return
        endif

        time_emis_fact(time_prof, row) = fact
    enddo
    close(emisUnit)
 enddo

end subroutine read_time_emis




! --------------------------------------------------------------------------------------------------------------------------
subroutine read_ruzice(path_file, ierror, error_msg)
! nacte ze souboru ruzice_file ruzici
! predpoklad: posledni sloupec CALM, 11 rozptyl. podminek

 implicit none

 character(*), intent(in   )	       	:: path_file
 integer,      intent(  out)  	       	:: ierror
 character(*), intent(  out)	       	:: error_msg
 real(dp), dimension(:,:), allocatable 	:: ruzice_in, ruzice_noCalm
 real(dp), dimension(:), allocatable	:: line
 character(200)				:: path_file_name
 character(200)				:: head
 integer				:: ruzUnit, ios, r, ai, poc_smeru_ruz, met, ind
 real(dp)				:: phi

 if (allocated(ruzice)) deallocate(ruzice); allocate(ruzice(nphi, poc_rozpt_t, met_num))

 error_msg = 'ruzici'
 ! nacti data
 ind = index(path_file, '.', back = .true.)
 do met = 1, met_num
    if (trim(met_name(met)) == '') then
       path_file_name = path_file
    else
       path_file_name = path_file(:ind-1) // '_' // trim(met_name(met)) // path_file(ind:)
    endif
    call check_file_exist(trim(path_file_name), ierror, error_msg)
    if (ierror /= 0) return

    open(unit = newunit(ruzUnit), file = trim(path_file_name), action = 'read')
     read(ruzUnit, * , iostat = ios) head;  head = trim(head)
     do while (head(1:1) == '!')
         read(ruzUnit, * , iostat = ios) head;  head = trim(head)
     enddo
     backspace(ruzUnit)

    read(ruzUnit, * , iostat = ios) poc_smeru_ruz

    if (allocated(ruzice_in)) deallocate(ruzice_in); allocate(ruzice_in(poc_smeru_ruz + 1, poc_rozpt_t))
    if (allocated(ruzice_noCalm)) deallocate(ruzice_noCalm);allocate(ruzice_noCalm(poc_smeru_ruz, poc_rozpt_t))
    if (allocated(line)) deallocate(line); allocate(line(poc_smeru_ruz + 1))

    do r = 1, poc_rozpt_t
        read(ruzUnit, * , iostat = ios) line

        if (ios /= 0) then
            call error_reading(path_file_name, r, ios, ierror, error_msg)
            close(ruzUnit)
            return
        endif
        ruzice_in(:, r) = line
    enddo
    close(ruzUnit)

    call rozpocet_calm(poc_rozpt_t, poc_smeru_ruz, ruzice_in, ruzice_noCalm)

    do r = 1, poc_rozpt_t
       do ai = 1, nphi
           phi = (ai - 1) * phi_step 
           ruzice(ai, r, met) = f_phi(ruzice_noCalm(:, r), phi, poc_smeru_ruz, phi_step)
       enddo
    enddo

    if (allocated(ruzice_in)) deallocate(ruzice_in)
    if (allocated(ruzice_noCalm)) deallocate(ruzice_noCalm)
    if (allocated(line)) deallocate(line)
 enddo

end subroutine read_ruzice



! -------------------------------------------------------------------------------------------------------------------------------
real(dp) function f_phi(ruzice, phi, poc_smeru, phi_step)
 ! spocte relativni cetnost z dane ruzice pro dany smer (rovnice 3.56), obecne pro jakykoliv uhlovy krok

 implicit none

 integer		:: poc_smeru
 real(dp)		:: ruzice(poc_smeru), phi, phi1, phi2, f_phi1, f_phi2, uhel_krok, pomer, phi_step

 uhel_krok = 360._dp / poc_smeru
 pomer = uhel_krok / phi_step                ! pomer kroku uhlu v zadane ruzici ku kroku vypocetnimu
 phi1 = int(phi / uhel_krok) * uhel_krok
 phi2 = phi1 + uhel_krok
 if (phi2 == 360) phi2 = 0 
 f_phi1 = ruzice(int(phi1 / uhel_krok) + 1)
 f_phi2 = ruzice(int(phi2 / uhel_krok) + 1)
 f_phi = (f_phi1 + (phi - phi1) * (f_phi2 - f_phi1) / uhel_krok) / (pomer * 100)

end function f_phi



! -------------------------------------------------------------------------------------------------------------------------------

subroutine rozpocet_calm(poc_rozpt_t, pocet_smeru, ruzice_in, ruzice_out)
 ! rozpocte CALM do vsech smeru v dane tride stability a rychlosti
 ! predpoklad, ze CALM je uveden u nejnizsi rychlosti, jinde jsou nuly

 implicit none

 integer		:: poc_rozpt_t, pocet_smeru, rt
 real(dp)		:: ruzice_in(pocet_smeru + 1, poc_rozpt_t), ruzice_out(pocet_smeru, poc_rozpt_t), podil

 ruzice_out = ruzice_in(1:pocet_smeru, :)
 do rt = 1, poc_rozpt_t
    podil = ruzice_in(pocet_smeru + 1, rt) / pocet_smeru
    ruzice_out(:, rt) = ruzice_out(:, rt) + podil
 enddo

end subroutine rozpocet_calm




! -------------------------------------------------------------------------------------------------------------------------------
subroutine read_ref_points(path_file, ierror, error_msg)
! nacte ze souboru ref_body_file referencni body, alokuje pole referencnich bodu a naplni ho

 implicit none

 character(*), intent(in   )	:: path_file
 integer,      intent(  out)    :: ierror
 character(*), intent(  out)	:: error_msg	
 character(20)			:: id
 character(200)		        :: head
 integer			:: refUnit, num_rows, ios, row
 real(dp)			:: x, y, z_user, l

 error_msg = 'referencnimi body'
 call check_file_exist(path_file, ierror, error_msg)
 if (ierror /= 0) return

 ! nejdrive projde soubor a spocte pocet radku
 num_rows = 0
 open(unit = newunit(refUnit), file = trim(path_file), action = 'read')
  read(refUnit, * , iostat = ios) head;  head = trim(head)
  do while (head(1:1) == '!')
      read(refUnit, * , iostat = ios) head;  head = trim(head)
  enddo
  backspace(refUnit)

 do 
     read(refUnit, * , iostat = ios) y
     if (ios < 0) exit
     num_rows = num_rows + 1
 enddo

 if (allocated(ref_body)) deallocate(ref_body)
 allocate(ref_body(num_rows))
 
 ! projde soubor znovu a nacte hodnoty
 rewind(refUnit)
  read(refUnit, * , iostat = ios) head;  head = trim(head)
  do while (head(1:1) == '!')
      read(refUnit, * , iostat = ios) head;  head = trim(head)
  end do
  backspace(refUnit)

 do row = 1, num_rows
     read(refUnit, * , iostat = ios) id, x, y, z_user, l
     if (ios /= 0) then
         call error_reading(path_file, row, ios, ierror, error_msg)
         close(refUnit)
         return
     endif
   
     ! test zda bod neni mimo sit
     if ((x.le.xllcorner) .or. (y.le.yllcorner) .or. (x.ge.xllcorner_upper) .or. (y.ge.yllcorner_upper)) then
         ierror = 4
         error_msg = 'Referencni bod ' // trim(id) // ' lezi mimo oblast vyskopisu.'
         close(refUnit)
         return
     endif

     ref_body(row)%id = id
     ref_body(row)%x = x
     ref_body(row)%y = y
     ref_body(row)%z = z_user
     ref_body(row)%l = l

 enddo
 close(refUnit)

end subroutine read_ref_points




! -------------------------------------------------------------------------------------------------------------------------------
subroutine read_sources(path_file1, path_file2, ierror, error_msg)
! nacte ze souboru zdroje_file a zdroje_popis_file udaje o zdrojich, alokuje a naplni pole zdroje

 implicit none

 character(*), intent(in   )	:: path_file1, path_file2	
 integer,      intent(  out)    :: ierror
 character(*), intent(  out)	:: error_msg
 character(20)			:: id
 character(10)			:: num
 character(3)			:: met
 character(3)			:: time_prof
 character(200)			:: head
 integer			:: zdrUnit, num_rows, ios, row, typ, shl, sk, ind, chl
 real(dp)			:: x, y, z_user, m, m_no, h, V, t, d, w, a, y0, AB(2), pd

 ! nacita definici zdroju -------------------------------------------------
 error_msg = 'definici zdroju'
 call check_file_exist(path_file1, ierror, error_msg)
 if (ierror /= 0) return

 ! nejdrive projde soubor a spocte pocet radku
 num_rows = 0
 open(unit = newunit(zdrUnit), file = trim(path_file1), action = 'read')
  read(zdrUnit, * , iostat = ios) head;  head = trim(head)
  do while (head(1:1) == '!')
      read(zdrUnit, * , iostat = ios) head;  head = trim(head)
  enddo
  backspace(zdrUnit)

 do 
     read(zdrUnit, * , iostat = ios) y
     if (ios < 0) exit
     num_rows = num_rows + 1
 enddo

 if (allocated(zdroje)) deallocate(zdroje)
 allocate(zdroje(num_rows))
 
 ! projde soubor znovu a nacte hodnoty
 rewind(zdrUnit)
  read(zdrUnit, * , iostat = ios) head;  head = trim(head)
  do while (head(1:1) == '!')
      read(zdrUnit, * , iostat = ios) head;  head = trim(head)
  enddo
  backspace(zdrUnit)

 do row = 1, num_rows
     if (is_time) then
	 read(zdrUnit, * , iostat = ios) id, x, y, z_user, typ, shl, sk, met, time_prof
     else
         read(zdrUnit, * , iostat = ios) id, x, y, z_user, typ, shl, sk, met
     endif
     if (ios /= 0) then
         call error_reading(path_file1, row, ios, ierror, error_msg)
         close(zdrUnit)
         return
     endif

     zdroje(row)%id = id

     if (typ == 3) then
        zdroje(row)%x1 = x
        zdroje(row)%y1 = y
     else
        zdroje(row)%x = x
        zdroje(row)%y = y
     endif
     zdroje(row)%z = z_user
     zdroje(row)%typ = typ
     zdroje(row)%shluk = shl
     zdroje(row)%skupina = sk

     if (locate_string(met_name, met) .eq. -1) then
        met_num = met_num + 1
        met_name(met_num) = met
        zdroje(row)%met = met_num
     else
        zdroje(row)%met = locate_string(met_name, met)
     endif     

     if (is_time) then
     	if (locate_string(time_prof_name, time_prof) .eq. -1) then
     	   time_prof_num = time_prof_num + 1
           time_prof_name(time_prof_num) = time_prof
           zdroje(row)%time_prof = time_prof_num
        else
           zdroje(row)%time_prof = locate_string(time_prof_name, time_prof)
        endif
     endif

 enddo
 close(zdrUnit)


 ! nacita popis zdroju -------------------------------------------------
 error_msg = 'popisem zdroju'
 call check_file_exist(path_file2, ierror, error_msg)
 if (ierror /= 0) return

 open(unit = newunit(zdrUnit), file = trim(path_file2), action = 'read')
  read(zdrUnit, * , iostat = ios) head;  head = trim(head)
  do while (head(1:1) == '!')
      read(zdrUnit, * , iostat = ios) head;  head = trim(head)
  enddo
  backspace(zdrUnit)

 do row = 1, num_rows
     read(zdrUnit, * , iostat = ios) id, m, m_no, h, V, t, d, w, a, pd, y0, chl
     if (ios /= 0) then
         call error_reading(path_file2, row, ios, ierror, error_msg)
         close(zdrUnit)
         return
     endif 
   
     ind = row ! locate_string(zdroje%id, id) ! predpokladam, ze jsou razeny stejne
     if (zdroje(ind)%id /= id) then
        ierror = -1
        write(num,'(I10)') row 
        error_msg = 'V radku ' // num // ' v souboru ' // trim(path_file2) // ' ocekavan zdroj id ' // trim(zdroje(ind)%id) // &
                    ' nalezen popis zdroje ' // trim(id) // '.'
        return
     endif
     zdroje(ind)%Mz = m
     zdroje(ind)%Mz_NO = m_no
     zdroje(ind)%H = h
     zdroje(ind)%Vs = V
     zdroje(ind)%ts = t
     zdroje(ind)%d = d
     zdroje(ind)%w0 = w  
     zdroje(ind)%alfa = a
     zdroje(ind)%Pd = pd
     zdroje(ind)%y0 = y0
     if (zdroje(ind)%typ == 3) then
         zdroje(ind)%ts = 0
         zdroje(ind)%w0 = 0  
         zdroje(ind)%d = 0
         zdroje(ind)%x2 = V
         zdroje(ind)%y2 = t  
         zdroje(ind)%x0 = d
         zdroje(ind)%z0 = w
         zdroje(ind)%x = (zdroje(ind)%x1 + zdroje(ind)%x2) / 2
         zdroje(ind)%y = (zdroje(ind)%y1 + zdroje(ind)%y2) / 2
         zdroje(ind)%y0 = sqrt((zdroje(ind)%x2 - zdroje(ind)%x1)**2 + (zdroje(ind)%y2 - zdroje(ind)%y1)**2)
     endif
     ! doplni parametry zdroje Q, A, B, beta
     zdroje(ind)%beta = get_beta(zdroje(ind)%ts)
     ! kontrola zda byl spravne zadan typ vypoctu pro Q
     if ((tep_vyd_type .ne. 'metodika') .and. (tep_vyd_type .ne. 'vyzk-zpr')) then
         ierror = 5
         error_msg = 'Hodnota parametru tep_vyd_type = ' // trim(tep_vyd_type) // ' je neplatna.'
         return
     endif
     zdroje(ind)%Q = tepel_vyd(zdroje(ind)%Vs, zdroje(ind)%ts, 0._dp)
     AB = QAB(zdroje(ind)%Q)
     zdroje(ind)%A = AB(1)
     zdroje(ind)%B = AB(2)
     if (chl == 1) then
         zdroje(ind)%chl_v = .true.
     else
         zdroje(ind)%chl_v = .FALSE.
     endif

     if ((zdroje(ind)%typ == 3) .or. (zdroje(ind)%typ == 2)) &
         zdroje(ind)%Vs = 0

     ! test zda zdroj neni mimo sit
     if ((zdroje(ind)%x.le.xllcorner) .or. (zdroje(ind)%y.le.yllcorner) &
          .or. (zdroje(ind)%x.ge.xllcorner_upper) .or. (zdroje(ind)%y.ge.yllcorner_upper)) then
         ierror = 4
         error_msg = 'Zdroj ' // trim(id) // ' lezi mimo oblast vyskopisu.'
         close(zdrUnit)
         return
     endif
 enddo
 close(zdrUnit)

 ! kontrola zda alfa neni 0 mozna pak bude neco divneho delat vypocet doby prekroceni

 ! serad zdroje dle alfa pro vypocet prekroceni
 if (is_exceed) call sort_sources(num_rows)
 
end subroutine read_sources



! -------------------------------------------------------------------------------------------------------------------------------
subroutine read_terrain_definition(path_file, ierror, error_msg)
! nacte ze souboru vyskopis_file definici site: ncols, nrows, xllcorner, yllcorner, cellsize
! spocte xllcorner_upper, yllcorner_upper

 implicit none

 character(*), intent(in   )	:: path_file
 integer,      intent(  out)    :: ierror
 character(*), intent(  out)	:: error_msg
 integer			:: terUnit, ios
 character(50)			:: pom	

 error_msg = 'terenem'

 call check_file_exist(path_file, ierror, error_msg)
 if (ierror /= 0) return

 open(unit = newunit(terUnit), file = trim(path_file), action = 'read')

 read(terUnit, *, iostat = ios) pom, ncols
 read(terUnit, *, iostat = ios) pom, nrows
 read(terUnit, *, iostat = ios) pom, xllcorner
 read(terUnit, *, iostat = ios) pom, yllcorner
 read(terUnit, *, iostat = ios) pom, cellsize
 read(terUnit, *, iostat = ios) pom, nodata_value

 close(terUnit) 

 if (ios /= 0) then
    call error_reading(path_file, 0, ios, ierror, error_msg)
    return
 endif 

 xllcorner_upper = xllcorner + cellsize * ncols
 yllcorner_upper = yllcorner + cellsize * nrows

end subroutine read_terrain_definition



! -------------------------------------------------------------------------------------------------------------------------------
subroutine read_terrain(path_file, ierror, error_msg)
! nacte ze souboru vyskopis_file nadmorske vysky, alokuje prislusne pole a naplni ho
! nacte z poli ref_body a zdroje maximalni rozsah vypocetni oblasti -> nacita a uklada jen nutny vyrez z vyskopisu
! predpoklada, ze uz byla zavolana subroutina read_terrain_definition, byla tedy zkontrolovana existence souboru


 implicit none

 character(*), intent(in   )		:: path_file
 integer,      intent(  out)  		:: ierror
 character(*), intent(  out)		:: error_msg
 integer				:: terUnit, ios, row
 real(dp)				:: xmin, ymin, xmax, ymax
 integer(2), dimension(:), allocatable	:: line	! pomocna promenna pro nacteni celeho radku
 integer				:: col_from, col_to, row_from, row_to  ! indexy orezane site
 character(50)				:: pom	

 open(unit = newunit(terUnit), file = trim(path_file), action = 'read')

 read(terUnit, *, iostat = ios) pom, ncols
 read(terUnit, *, iostat = ios) pom, nrows
 read(terUnit, *, iostat = ios) pom, xllcorner
 read(terUnit, *, iostat = ios) pom, yllcorner
 read(terUnit, *, iostat = ios) pom, cellsize
 read(terUnit, *, iostat = ios) pom, nodata_value

 if (ios /= 0) then
    call error_reading(path_file, 0, ios, ierror, error_msg)
    close(terUnit)
    return
 endif 

 ! zjisti jakou oblast vyskopisu potrebuji nacist
 xmin = minval(ref_body%x)
 xmax = maxval(ref_body%x)
 if (minval(zdroje%x).lt.xmin) xmin = minval(zdroje%x)
 if (maxval(zdroje%x).gt.xmax) xmax = maxval(zdroje%x)
 ymin = minval(ref_body%y)
 ymax = maxval(ref_body%y)
 if (minval(zdroje%y).lt.ymin) ymin = minval(zdroje%y)
 if (maxval(zdroje%y).gt.ymax) ymax = maxval(zdroje%y)

 col_from = terrain_ind(xmin, xllcorner, cellsize)
 col_to = terrain_ind(xmax, xllcorner, cellsize)
 row_from = nrows - terrain_ind(ymax, yllcorner, cellsize) + 1
 row_to = nrows - terrain_ind(ymin, yllcorner, cellsize) + 1

 if (allocated(z)) deallocate(z)
 allocate(z(col_to - col_from + 1 , row_to - row_from + 1))
 if (allocated(line)) deallocate(line)
 allocate(line(ncols))
 ! kontrola na nodata_value
 do row = 1, row_from - 1
     read(terUnit, * , iostat = ios) 
     if (ios /= 0) then
         call error_reading(path_file, row, ios, ierror, error_msg)        
         close(terUnit)
         return
     endif
 enddo
 do row = row_from, row_to
     read(terUnit, * , iostat = ios) line     
     if (ios /= 0) then
         call error_reading(path_file, row, ios, ierror, error_msg)        
         close(terUnit)
         return
     endif
     z(:, row_to - row + 1) = line(col_from:col_to)
 enddo
 close(terUnit)

 ! zapise nove hodnoty xllcorner, yllcorner
 xllcorner_upper = xllcorner + cellsize * col_to
 yllcorner_upper = yllcorner + cellsize * (nrows - row_from + 1)
 xllcorner = xllcorner + cellsize * (col_from - 1)
 yllcorner = yllcorner + cellsize * (nrows - row_to)

end subroutine read_terrain


! -------------------------------------------------------------------------------------------------------------------------------
subroutine dopln_z
! do poli ref_body a zdroje doplni z-souradnici z vyskopisu

 implicit none

 integer   :: i, pocet_ref_bodu, pocet_zdroju, xind, yind

 pocet_ref_bodu = size(ref_body)
 do i = 1, pocet_ref_bodu
    xind = terrain_ind(ref_body(i)%x, xllcorner, cellsize)
    yind = terrain_ind(ref_body(i)%y, yllcorner, cellsize)
    ref_body(i)%z = z(xind, yind)
 enddo

 pocet_zdroju = size(zdroje)
 do i = 1, pocet_zdroju
    xind = terrain_ind(zdroje(i)%x, xllcorner, cellsize)
    yind = terrain_ind(zdroje(i)%y, yllcorner, cellsize)
    zdroje(i)%z = z(xind, yind)
 enddo
end subroutine dopln_z



! -------------------------------------------------------------------------------------------------------------------------------
subroutine read_chlad_vez(path_file, ierror, error_msg)
! nacte ze souboru path_file klimaticke udaje pro chladici veze

 implicit none

 character(*), intent(in   )	:: path_file
 integer,      intent(  out)    :: ierror
 character(*), intent(  out)	:: error_msg
 character			:: pom
 integer			:: chlUnit, row, ios, ci, ui

 error_msg = 'klimatickymi udaji pro chladici veze'
 call check_file_exist(path_file, ierror, error_msg)
 if (ierror /= 0) return

 open(unit = newunit(chlUnit), file = trim(path_file), action = 'read')

 read(chlUnit, *) pom
 do ci = 1, poc_trid_st
      read(chlUnit, *) pom
      read(chlUnit, * , iostat = ios) grt(:, :, ci)
 enddo

 read(chlUnit, *) pom
 read(chlUnit, * , iostat = ios) frt

 do ui = 1, poc_trid_u
      read(chlUnit, *) pom
      read(chlUnit, * , iostat = ios) Vch1(:, :, ui)
 enddo

 do ui = 1, poc_trid_u
      read(chlUnit, *) pom
      read(chlUnit, * , iostat = ios) Tch1(:, :, ui)
 enddo

 if (ios /= 0) then
     call error_reading(path_file, row, ios, ierror, error_msg)
     close(chlUnit)
     return
 endif 
 
end subroutine read_chlad_vez


! -------------------------------------------------------------------------------------------------------------------------------
subroutine read_output_ref_p(path_file, ierror, error_msg)
! nacte ze souboru output_ref_p id ref. bodu, ktere maji byt na hodinovem vystupu

 implicit none

 character(*), intent(in   )	:: path_file
 integer,      intent(  out)    :: ierror
 character(*), intent(  out)	:: error_msg	
 character(20)			:: id
 integer			:: refUnit, ios, ind	

 error_msg = 'vystupnimi referencnimi body'
 call check_file_exist(path_file, ierror, error_msg)
 if (ierror /= 0) return

 ref_body%output = .FALSE.
 open(unit = newunit(refUnit), file = trim(path_file), action = 'read')
 do 
     read(refUnit, * , iostat = ios) id
     if (ios < 0) exit
     ind = locate_string(ref_body%id, id)    
     if (ind .eq. -1) ierror = ind
     if (ierror /= 0) then
        error_msg = 'Referencni bod ' // trim(id) // ' nachazejici se v souboru ' // trim(path_file) // ' nebyl definovan.'
        return
     endif
     ref_body(ind)%output = .TRUE.
 enddo
 close(refUnit)

end subroutine read_output_ref_p



! -------------------------------------------------------------------------------------------------------------------------------
subroutine sort_sources(poc_zdroju)
 ! seradi zdroje podle alfa od nejvyssiho po nejnizsi pro ucely vypoctu dob prekroceni, kap 3.3.3

 implicit none

 integer		:: si, poc_zdroju, imax
 type(zdroj)		:: zdroje_p(poc_zdroju)     ! pomocna promenna pro preskladani zdroju

 do si = 1, poc_zdroju
     imax =  maxloc(zdroje%alfa, 1)
     zdroje_p(si) = zdroje(imax)
     zdroje(imax)%alfa = -1
 enddo

 zdroje = zdroje_p

end subroutine




! -------------------------------------------------------------------------------------------------------------------------------
function QAB(Q)
    ! vrati hodnoty parametru A a B dle tepelne vydatnosti zdroje
    ! Metodika SYMOS, tab. 3.2
    implicit none
    
    real(dp), intent(in   ) :: Q     ! [MW]  tepelna vydatnost zdroje
    real(dp), dimension(2)  :: QAB  

    if (Q >= 20) then
        QAB(1) = 30
        QAB(2) = 0.7
    else 
        QAB(1) = 90
        QAB(2) = 1/3._8
    end if
end function QAB



! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function tepel_vyd(Vs, ts, t0)
    ! vrati tepelnou vydatnost v [MW] 
    ! Metodika SYMOS vztah (2.10)  
    ! Upraveno dle vyzkumne zpravy v souladu s IDEou
    implicit none
    
    real(dp), intent(in   ) :: Vs     ! [Nm3/s] objemovy tok spalin za normalnich podminek (0°C, 101325Pa)
    real(dp), intent(in   ) :: ts, t0 ! [°C]  je teplota spalin nebo vzdušniny v koruně komína nebo výduchu; t0 ... okolni teplota

    tepel_vyd = 1e-3 * Vs * 1.371 * (ts - t0) 

    if ((ts <= 80) .and. (tep_vyd_type == 'vyzk-zpr')) tepel_vyd = 1e-3 * Vs * 1.004 * (ts - t0) 

end function tepel_vyd


! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function get_beta(ts)
    ! urci hodnoty parametru BETA podle vystupni teploty spalin
    ! Metodika SYMOS vztah (3.23)
    implicit none
    
    real(dp), intent(in) :: ts ! [°C]  je teplota spalin nebo vzdušniny v koruně komína nebo výduchu

    get_beta = 1
    if (ts<=30) then
        get_beta = 0
    else if (ts<80) then
        get_beta = (ts-30.)/50.
    end if
end function get_beta


! -------------------------------------------------------------------------------------------------------------------------------
integer function terrain_ind(sour, sourcorner, cellsize)
 ! ze souradnice bodu sour, souradnice rohu sourcorner a velikosti bunky urci index bunky, ve kterem se bod nachazi (1D)

 implicit none

 real(dp), intent(in   ) :: sour, sourcorner, cellsize

 terrain_ind = int(abs(sour - sourcorner) / cellsize) + 1

end function terrain_ind


! -------------------------------------------------------------------------------------------------------------------------------
subroutine check_file_exist(file_name, ierror, error_msg)
 ! Zkontroluje zda existuje soubor file_name (ierror = 0)
 ! navic rozlisi zda nebyl zadan (ierror = 1) nebo neexistuje (ierror = 2)
 ! pokud nastala chyba, do error_msg doplni hlasku

 implicit none
 
 character(*)		:: file_name
 character(*)		:: error_msg
 integer		:: ierror, ind
 logical 		:: is_file

 ierror = 0

 ind = index(file_name, '/', back = .true.)
 if (trim(file_name(ind+1:)) == '') then
   ierror = 1
   error_msg = 'Soubor s '// trim(error_msg) // ' nebyl zadan.'
   return
 endif

 inquire(file = file_name, exist = is_file)  
 if (.not. is_file) then
     ierror = 2
     error_msg = 'Soubor ' // trim(file_name) // ' s '// trim(error_msg) // ' neexistuje.'
     return
 endif

end subroutine check_file_exist



! -------------------------------------------------------------------------------------------------------------------------------
subroutine check_directory_exist(dir_name, ierror, error_msg)
 ! Zkontroluje zda existuje adresar dir_name
 ! pokud nastala chyba, do error_msg doplni hlasku

 implicit none
 
 character(*)		:: dir_name
 character(*)		:: error_msg
 integer		:: ierror, refUnit
 logical 		:: is_file

 ierror = 0

 inquire(file = trim(dir_name) // 'test_file', exist = is_file)  
 if (is_file) return
 open(unit = newunit(refUnit), file = trim(dir_name) // 'test_file', iostat = ierror)
 if (ierror /= 0) then
    error_msg = 'Problem se zapisem do ' // trim(error_msg) // ': ' // trim(dir_name) // &
                '. Zkontrolujte ze adresar existuje.' 
    return
 endif
 close(refUnit, status = 'delete')

end subroutine check_directory_exist


! -------------------------------------------------------------------------------------------------------------------------------
subroutine error_reading(path_file, row, ios, ierror, error_msg)
 ! vytvori chybovou hlasku error_msg, pokud nastala chyba ios v prubehu nacitani dat na radku row (ierror = 3)

 implicit none
 
 character(*), intent(in   )	:: path_file
 integer,      intent(in   )	:: row, ios
 integer,      intent(  out)	:: ierror
 character(*), intent(  out)	:: error_msg
 integer			:: ind
 character(10)			:: num

 ierror = 3
 write(num,'(I10)') row 
 ind = index(path_file, '/', back = .true.)
 error_msg = 'Chyba ctení dat ze souboru ' // trim(path_file(ind+1:)) // '.'
 if (ios < 0) then
    error_msg = trim(error_msg) // new_line('A') // 'Zkontrolujte zda v nekterych radcich nechybi data.'
 else
    error_msg = trim(error_msg) // new_line('A') // 'Chyba na radku ' // trim(adjustl(num)) // '.'
 endif

end subroutine error_reading


! -------------------------------------------------------------------------------------------------------------------------------
subroutine error_nml_read(path_file, ios, ierror, error_msg)
 ! vytvori chybovou hlasku error_msg, pokud nastala chyba ios v prubehu nacitani nml dat 

 implicit none
 
 character(*), intent(in   )	:: path_file
 integer,      intent(in   )	:: ios
 integer,      intent(  out)	:: ierror
 character(*), intent(  out)	:: error_msg
 integer			:: ind

 ierror = 3
 ind = index(path_file, '/', back = .true.)
 error_msg = 'Chyba ctení dat ze souboru ' // trim(path_file(ind+1:)) // '.'

 if (ios < 0) then
    error_msg = trim(error_msg) // new_line('A') // 'Neocekavany konec souboru.' // new_line('A') // & 
                  'Zkontrolujte zda mate na zacatku souboru retezec &init a na konci znak /.'
 else
    error_msg = trim(error_msg) // new_line('A') // 'Zkontrolujte zda jsou nazvy parametru uvedeny spravne.'
 endif

end subroutine error_nml_read


! -------------------------------------------------------------------------------------------------------------------------------
subroutine raise_error(error_msg)
 ! zahlasi error_msg a ukonci program

 implicit none

 character(*)		:: error_msg

 write(*,*) trim(error_msg)
 stop

end subroutine raise_error


! -------------------------------------------------------------------------------------------------------------------------------

integer function locate_string(array, string)
 ! najde string v poli array, pokud tam neni vrati -1
 implicit none
 character(*), dimension(:)	:: array
 character(*)			:: string
 integer			:: i
 
 locate_string = -1
 do i = 1, size(array)
     if (array(i) == string) then
         locate_string = i
         return
     endif
 enddo

end function locate_string



! ---------------------------------------------------------------------------------------------------------
integer function count_items(s1) 
 ! Count items in string that are blank separated
 ! from module String_Functions by David Frank
 implicit none
 character(*) 	     :: s1
 character(Clen(s1)) :: s
 integer 	     :: i, k, lens

 s = s1                   ! remove possible last char null
 k = 0  
 if (s /= ' ') k = 1      ! string has at least 1 item
 lens = LEN_TRIM(s)
 do i = 1, lens - 1
    if ((s(i:i)/=' ').and.(s(i:i)/=',').and.(s(i:i)/=';').and.(s(i:i)/="	") &  
        .and.((s(i+1:i+1)==' ').or.(s(i+1:i+1) == ',').or.(s(i+1:i+1)==';').or.(s(i+1:i+1) == "	"))) k = k + 1
 enddo
 if ((s(lens:lens)==' ').or.(s(lens:lens) == ',').or.(s(lens:lens)==';').or.(s(lens:lens) == "	")) k = k - 1
 count_items = k

end function count_items



! ------------------------
pure integer function Clen(s)      
 ! returns same result as LEN unless:
 ! last non-blank char is null
 ! from module String_Functions by David Frank
 implicit none
 character(*), intent(in) :: s      ! last non-blank char is null
 integer :: i

 Clen = LEN(s)
 i = LEN_TRIM(s)
 if (s(i:i) == CHAR(0)) Clen = i-1  ! len of C string
end function Clen


! -------------------------------------------------------------------------------------------------------------------------------
integer function newunit(unit)
! This is a simple function to search for an available unit.
! LUN_MIN and LUN_MAX define the range of possible LUNs to check.
! The unit value is returned by the function, and also by the optional
! argument. This allows the function to be used directly in an open
! statement, and optionally save the result in a local variable.
! If no units are available, -1 is returned.
  integer, intent(out), optional :: unit
! local
  integer, parameter :: LUN_MIN = 10, LUN_MAX = 1000
  logical :: opened
  integer :: lun
! begin
  newunit = -1
  do lun=LUN_MIN,LUN_MAX
    inquire(unit=lun, opened=opened)
    if (.not. opened) then
      newunit=lun
      exit
    end if
  end do
  if (present(unit)) unit=newunit
end function newunit


! -------------------------------------------------------------------------------------------------------------------------------
!> Přidá oddělovač souborů na konec cesty idir pokud tam už nejsou.
!> -------------------------------------------------
subroutine add_sep2dir(idir)

 implicit none

 character(250)		:: idir
 integer		:: ind

 ind = lnblnk(idir)
 if ((idir(ind:ind) /= '/').and.(idir(ind:ind) /= '\')) then
    idir = trim(idir) // '/'
 end if

end subroutine add_sep2dir

end module mod_init
